<?php

namespace bitcream\gmvricambi\custom\backend\controllers;

use Yii;
use bitcream\gmvricambi\custom\backend\models\Order;
use bitcream\gmvricambi\custom\backend\models\GMVCountry;
use bitcream\gmvricambi\custom\backend\models\GMVProvince;
use bitcream\gmvricambi\custom\backend\models\GMVShippingAddress;
use bitcream\gmvricambi\custom\backend\models\OrderCustomerSearch;
use bitcream\gmvricambi\custom\backend\models\OrderItemSearch;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\filters\AccessControl;
use bitcream\gmvricambi\custom\common\models\CustomerUser;
use bitcream\gmvricambi\custom\common\models\GMVOrder;

/**
 * OrderController implements the CRUD actions for Order model.
 */
class OrderCustomerController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => ['login', 'error'],
                        'allow' => true,
                    ],
                    [
                        'actions' => ['update', 'delete', 'view'],
                        'allow' => true,
                        'roles' => ['ViewOwnContent',],
                        'roleParams' => function() {
                            return ['content' => Order::findOne(['id' => Yii::$app->request->get('id')])];
                        },
                    ],
                    [
                        'actions' => ['index', 'create', 'delete', 'delete', 'activate-order', 'confirm','sent-gmv', 'pay'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }


    /**
     * Lists all Order models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new OrderCustomerSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);
        // Yii::$app->user->id;
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $order = Order::findOne(['customer_cod' => $customerUser->customerCod]);
        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
            'customerUser' => $customerUser,
        ]);
    }

    /**
     * Displays a single Order model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id)
    {
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $order = Order::findOne(['customer_cod' => $customerUser->customerCod]);

        $searchModelOrderItem = new OrderItemSearch();
        $dataProviderOrderItem = $searchModelOrderItem->search(Yii::$app->request->queryParams);

        return $this->render('view', [
            'model' => $this->findModel($id),
            'searchModelOrderItem' => $searchModelOrderItem,
            'dataProviderOrderItem' => $dataProviderOrderItem,
            'customerUser' => $customerUser,
        ]);
    }

     /**
     * Creates a new Order model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new Order();
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $order = Order::findOne(['customer_cod' => $customerUser->customerCod]);
        if ($model->load(Yii::$app->request->post())) {
            $model->customer_cod = $customerUser->customerCod;
            $model->setOrderStatus("processing"); 
            $model->save();
           
            return $this->redirect(['index']);
        }

        return $this->render('create', [
            'model' => $model,
            'countries' => GMVCountry::find()->all(),
            'provinces' => GMVProvince::find()->all(),
            'shippingAddress' => GMVShippingAddress::find()->where(['customer_cod' => $customerUser->customerCod])->all(),
            'order' => GMVOrder::find()->all(),
        ]);
    }

    /**
     * Updates an existing Order model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $order = Order::findOne(['customer_cod' => $customerUser->customerCod]);
        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            return $this->redirect(['index']);
        }

        return $this->render('update', [
            'model' => $model,
            'countries' => GMVCountry::find()->all(),
            'provinces' => GMVProvince::find()->all(),
            'shippingAddress' => GMVShippingAddress::find()->where(['customer_cod' => $customerUser->customerCod])->all(),
            'order' => GMVOrder::find()->all(),
        ]);
    }

    public function actionDelete($id)
    {
        $model = $this->findModel($id);
        $model->setOrderStatus("deleted"); 
        $model->delete();
        return $this->redirect(['index']);
    }

    public function actionActivateOrder($id)
    {
        $model = $this->findModel($id);
        CustomerUser::getCurrentUser()->unsetActiveOrder($id);
        CustomerUser::getCurrentUser()->setActiveOrder($model);   
        return $this->redirect(['index']);
    }

    public function actionSentGmv($id)
    {
        $model = $this->findModel($id);     
        $model->setOrderStatus("sent"); 
        CustomerUser::getCurrentUser()->setClosedOrder($model);   
        return $this->redirect(['index']);
    }

    public function actionConfirm($id)
    {
        $model = $this->findModel($id);     
        $model->setOrderStatus("confirm"); 
        return $this->redirect(['index']);
    }

    public function actionPay($id)
    {
        $model = $this->findModel($id);     
        $model->setOrderStatus("paid"); 
        return $this->redirect(['index']);
    }

    /**
     * Finds the Order model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Order the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Order::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException(Yii::t('gmv', 'The requested page does not exist.'));
    }
}
