<?php

namespace bitcream\gmvricambi\custom\backend\controllers;

use Yii;
use bitcream\gmvricambi\custom\backend\models\Order;
use bitcream\gmvricambi\custom\backend\models\GMVCountry;
use bitcream\gmvricambi\custom\backend\models\GMVProvince;
use bitcream\gmvricambi\custom\backend\models\GMVShippingAddress;
use bitcream\gmvricambi\custom\backend\models\OrderSearch;
use bitcream\gmvricambi\custom\backend\models\OrderItemSearch;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\filters\AccessControl;
use bitcream\gmvricambi\custom\common\models\CustomerUser;
use bitcream\gmvricambi\custom\common\models\GMVCustomer;
use bitcream\gmvricambi\custom\common\behaviors\OrderStatusBehaviors;
use bitcream\gmvricambi\custom\common\models\GMVOrder;

/**
 * OrderController implements the CRUD actions for Order model.
 */
class OrderController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => ['login', 'error'],
                        'allow' => true,
                    ],
                    [
                        'actions' => ['create'],
                        'allow' => true,
                        'roles' => ['CreateAllOrder'],
                    ],
                    [
                        'actions' => ['update'],
                        'allow' => true,
                        'roles' => ['UpdateAllOrder'],
                    ],
                    [
                        'actions' => ['view'],
                        'allow' => true,
                        'roles' => ['ViewAllOrder'],
                    ],
                    [
                        'actions' => ['index', 'create', 'delete', 'delete','activate-add-product-order', 'reset-filter', 'set-filter', 'set-status', 'activate-order', 'confirm','sent-gmv', 'pay', 'index-processing', 'index-deleted', 'index-sent-gmv', 'index-confirmed-gmv', 'payment-success'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }

    /**
     * Lists all Order models.
     * @return mixed
     */
    public function actionIndex()
    {
        $customerUser = CustomerUser::findOne(Yii::$app->user->id); 
        $searchModel = new OrderSearch();        
        $searchModel->orderStatusFilter($customerUser->ordersStatusSelected);
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'customerUser' => $customerUser,
            'dataProvider' => $dataProvider,
            'orderStatusSelected' => $customerUser->getOrdersStatusSelected(),
            'status' => OrderStatusBehaviors::getOrderAdminStatus()
        ]);
    }

    /**
     * Displays a single Order model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id)
    {
        $searchModelOrderItem = new OrderItemSearch();
        $dataProviderOrderItem = $searchModelOrderItem->search(Yii::$app->request->queryParams);

        return $this->render('view', [
            'model' => $this->findModel($id),
            'searchModelOrderItem' => $searchModelOrderItem,
            'dataProviderOrderItem' => $dataProviderOrderItem,
        ]);
    }

    /**
     * Creates a new Order model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new Order();
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $model->order_date =  time();
        if ($model->load(Yii::$app->request->post())) {
            $model->postal_code = (string)Yii::$app->request->post()['Order']['postal_code'];
            $model->total = 0;        
            $model->save();
            return $this->redirect(['view', 'id' => $model->id]);
        }

        return $this->render('create', [
            'model' => $model,
            'customer' => GMVCustomer::find()->all(),
            'countries' => GMVCountry::find()->all(),
            'provinces' => GMVProvince::find()->all(),
            'shippingAddress' => GMVShippingAddress::find()->where(['customer_cod' => $customerUser->customerCod])->all(),
        ]);
    }

    /**
     * Updates an existing Order model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);

        if(!$model->order_date){
            $model->order_date =  time();
        }
        if ($model->load(Yii::$app->request->post())) {
            $model->postal_code = (string)Yii::$app->request->post()['Order']['postal_code'];
            $model->save();
            return $this->redirect(['index']);
        }

        return $this->render('update', [
            'model' => $model,
            'customer' => GMVCustomer::find()->all(),
            'countries' => GMVCountry::find()->all(),
            'provinces' => GMVProvince::find()->all(),
            'shippingAddress' => GMVShippingAddress::find()->where(['customer_cod' => $customerUser->customerCod])->all(),
        ]);
    }

    /**
     * Deletes an existing Order model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    public function actionSetStatus($status){
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $customerUser->setOrdersStatusSelected($status);
        return $this->redirect(['index']);
    }

    public function actionSetFilter($startDate, $endDate, $locality, $total, $customer){
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $customerUser->setOrderStartDateSelected($startDate);
        $customerUser->setOrderEndDateSelected($endDate);
        $customerUser->setOrderDestinationSelected($locality);
        $customerUser->setTotalSelected($total);
        $customerUser->setCustomerSelected($customer);
        return $this->redirect(['index']);
    }

    public function actionResetFilter(){
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $customerUser->unsetOrderStartDateSelected();
        $customerUser->unsetOrderEndDateSelected();
        $customerUser->unsetOrderDestinationSelected();
        $customerUser->unsetTotalSelected();
        $customerUser->unsetCustomerSelected();
        return $this->redirect(['index']);
    }

    /**
     * Finds the Order model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Order the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Order::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException(Yii::t('gmv', 'The requested page does not exist.'));
    }
}
