<?php

namespace bitcream\gmvricambi\custom\backend\controllers;

use Yii;
use bitcream\gmvricambi\custom\backend\models\Order;
use bitcream\gmvricambi\custom\backend\models\OrderItem;
use bitcream\gmvricambi\custom\backend\models\GMVCountry;
use bitcream\gmvricambi\custom\backend\models\GMVProvince;
use bitcream\gmvricambi\custom\backend\models\GMVCustomer;
use bitcream\gmvricambi\custom\backend\models\GMVShippingAddress;
use bitcream\gmvricambi\custom\backend\models\OrderItemSearch;
use bitcream\gmvricambi\custom\backend\models\GMVOrderSearch;
use bitcream\gmvricambi\custom\backend\models\OrderCustomerSearch;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use yii\filters\AccessControl;
use DateTime;
use bitcream\gmvricambi\custom\common\models\CustomerUser;
use bitcream\gmvricambi\custom\common\models\GMVOrder;
use bitcream\gmvricambi\custom\common\behaviors\OrderStatusBehaviors;
use bitcream\gmvricambi\custom\common\utils\Stripe;

/**
 * OrderController implements the CRUD actions for Order model.
 */
class OrderCustomerController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::className(),
                'rules' => [
                    [
                        'actions' => ['login', 'error'],
                        'allow' => true,
                    ],
                    [
                        'actions' => ['update', 'delete', 'view'],
                        'allow' => true,
                        'roles' => ['ViewOwnContent',],
                        'roleParams' => function() {
                            return ['content' => Order::findOne(['id' => Yii::$app->request->get('id')])];
                        },
                    ],
                    [
                        'actions' => ['index', 'create', 'delete', 'activate-add-product-order', 'reset-filter', 'set-filter', 'set-status', 'activate-order', 'confirm','sent-gmv', 'pay', 'index-processing', 'index-deleted', 'index-sent-gmv', 'index-confirmed-gmv', 'payment-success', 'update-quantity-product-to-order', 'duplicate-order'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::className(),
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }


    /**
     * Lists all Order models.
     * @return mixed
     */
    public function actionIndex()
    {
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);        
        $searchModel = OrderStatusBehaviors::getModelType($customerUser->ordersStatusSelected);        
        $searchModel->orderStatusFilter($customerUser->ordersStatusSelected);
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);
        $index = OrderStatusBehaviors::getIndex($customerUser->ordersStatusSelected);

        return $this->render($index, [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
            'customerUser' => $customerUser,
            'datatable' => OrderStatusBehaviors::getDatatableView($customerUser->ordersStatusSelected),
            'orderStatusSelected' => $customerUser->getOrdersStatusSelected()
         ]);
    }

    /**
     * Displays a single Order model.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($id)
    {
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $model = $this->findModel($id);
        $searchModelOrderItem = new OrderItemSearch();
        $dataProviderOrderItem = $searchModelOrderItem->search(Yii::$app->request->queryParams);

        return $this->render('view', [
            'model' => $this->findModel($id),
            'searchModelOrderItem' => $searchModelOrderItem,
            'dataProviderOrderItem' => $dataProviderOrderItem,
            'customerUser' => $customerUser,
            'status' => $model->order_status,
            'rating' => $customerUser->customer->checkRating(),
            'paymentMethods' => $model->customer->getPaymentMethods($model->total),
        ]);
    }

     /**
     * Creates a new Order model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $model = new Order();
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $order = Order::findOne(['customer_cod' => $customerUser->customerCod]);
        $model->order_date =  time();
        $model->delivery_date =  date('d/m/Y', strtotime("+1 week"));
        if ($model->load(Yii::$app->request->post())) {
            $model->business_name = (string)Yii::$app->request->post()['Order']['business_name'];
            $model->customer_cod = $customerUser->customerCod;
            $model->setOrderStatus("processing"); 
            CustomerUser::getCurrentUser()->setActiveOrder($model);   
            $model->postal_code = (string)Yii::$app->request->post()['Order']['postal_code'];
            $model->plant_reference = serialize(Yii::$app->request->post()['plant_code']);
            $model->total = 0; 
            $date = str_replace('/', '-', Yii::$app->request->post()['Order']['delivery_date']);
            $model->delivery_date = date('U', strtotime($date));
            $model->save();
           
            return $this->redirect(['view', 'id' => $model->id]);
        }

        return $this->render('create', [
            'model' => $model,
            'countries' => GMVCountry::find()->orderBy(['description' => SORT_ASC])->all(),
            'provinces' => GMVProvince::find()->orderBy(['description' => SORT_ASC])->all(),
            'shippingAddress' => GMVShippingAddress::find()->where(['customer_cod' => $customerUser->customerCod])->all(),
            'order' => GMVOrder::find()->all(),
        ]);
    }

    /**
     * Updates an existing Order model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $order = Order::findOne(['customer_cod' => $customerUser->customerCod]);
        $model->plant_reference = unserialize($model->plant_reference);
        if(!$model->order_date){
            $model->order_date =  time();
        }
        $model->delivery_date =  date('d/m/Y', $model->delivery_date);
        if ($model->load(Yii::$app->request->post())) {
            $model->postal_code = (string)Yii::$app->request->post()['Order']['postal_code'];
            $model->business_name = (string)Yii::$app->request->post()['Order']['business_name'];
            $model->plant_reference = serialize(Yii::$app->request->post()['plant_code']);
            $date = str_replace('/', '-', Yii::$app->request->post()['Order']['delivery_date']);
            $model->delivery_date = date('U', strtotime($date));
            $model->save();
            return $this->redirect(['view', 'id' => $model->id]);
        }

        return $this->render('update', [
            'model' => $model,
            'countries' => GMVCountry::find()->all(),
            'provinces' => GMVProvince::find()->all(),
            'shippingAddress' => GMVShippingAddress::find()->where(['customer_cod' => $customerUser->customerCod])->all(),
            'order' => GMVOrder::find()->all(),
        ]);
    }

    public function actionDelete($id)
    {
        $model = $this->findModel($id);
        $model->deleteOrder(); 
        $model->delete();
        return $this->redirect(['index']);
    }

    public function actionActivateOrder($id)
    {
        $model = $this->findModel($id);
        $model->activateOrder();
        return $this->redirect(['index']);
    }

    public function actionActivateAddProductOrder($id)
    {
        $model = $this->findModel($id);
        $model->activateOrder();
        return $this->redirect(['gmv-article-customer/index']);
    }

    public function actionSentGmv($id)
    {
        $model = $this->findModel($id);    
        $model->sendOrder(); 
        return $this->redirect(['index-sent-gmv']);
    }

    public function actionConfirm($id)
    {
        $model = $this->findModel($id);     
        $response = $model->confirm(Yii::$app->request->post()['payment_method'], $model->total);
        if($response == true){
            $redirect = Stripe::sendPayment($model); 
            return $this->redirect($redirect['url']);
        }                 
        return $this->redirect(['index']);
    }

    public function actionPaymentSuccess($id){
        $model = $this->findModel($id); 
        $model->payOrder();

        $this->redirect(['view', 'id' => $model->id]);
    }

    public function actionSetStatus($status){
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $customerUser->setOrdersStatusSelected($status);
        return $this->redirect(['index']);
    }

    public function actionSetFilter($startDate, $endDate, $locality){
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $customerUser->setOrderStartDateSelected($startDate);
        $customerUser->setOrderEndDateSelected($endDate);
        $customerUser->setOrderDestinationSelected($locality);
        return $this->redirect(['index']);
    }

    public function actionResetFilter(){
        $customerUser = CustomerUser::findOne(Yii::$app->user->id);
        $customerUser->unsetOrderStartDateSelected();
        $customerUser->unsetOrderEndDateSelected();
        $customerUser->unsetOrderDestinationSelected();
        return $this->redirect(['index']);
    }

    public function actionDuplicateOrder($id){
        $model = $this->findModel($id); 
        $model->duplicateOrder();
        $model->activateOrder();
        return $this->redirect(['index']);
    }

    public function actionUpdateQuantityProductToOrder(){

        $result = OrderItem::updateQuantityOrder(Yii::$app->request->post()['OrderItem']['id'], Yii::$app->request->post()['OrderItem']['quantity']);
        
        if($result){
            $response = Yii::$app->response;
            $response->format = \yii\web\Response::FORMAT_JSON;
            $response->data = $result;
            $response->statusCode = 200;
        }else{
            $response->data = false;
        }

        return $response;
    }
    
    public function actionDuplicateOrder($id){
        $model = $this->findModel($id); 
        $orderDuplicated = $model->duplicateOrder();
        $orderDuplicated->activateOrder();
        return $this->redirect(['update', 'id' => $orderDuplicated->id]);
    }

    /**
     * Finds the Order model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Order the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Order::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException(Yii::t('gmv', 'The requested page does not exist.'));
    }
}
