<?php
namespace bitcream\gmvricambi\custom\common\behaviors;

use yii\base\Behavior;
use bitcream\gmvricambi\custom\common\models\CustomerUser;
use bitcream\gmvricambi\custom\backend\models\GMVOrderSearch;
use bitcream\gmvricambi\custom\backend\models\GMVOrderOtherSearch;
use bitcream\gmvricambi\custom\backend\models\OrderCustomerSearch;

class OrderStatusBehaviors extends Behavior
{
    private $order_status;
    private $selectedOrderStatus;
    
    const ORDER_STATUS_PROCESSING = 'processing';
    const ORDER_STATUS_DELETED = 'deleted';
    const ORDER_STATUS_SENT = 'sent';    
    const ORDER_STATUS_PAID = 'paid';
    const ORDER_STATUS_GMV_RECEIVED = 'confirmed';
    const ORDER_STATUS_GMV_CONFIRMED = 'gmv_confirmed';
    const ORDER_STATUS_GMV_SENT = 'gmv_sent';
    const ORDER_STATUS_GMV_PROCESSING = 'gmv_processing';
    const ORDER_STATUS_GMV_READY = 'gmv_ready';
    const ORDER_STATUS_GMV_CANCEL = 'gmv_cancel';
    const ORDER_STATUS_GMV_OTHER = 'gmv_order_other';

    const EVENT_AFTER_ORDER_SEND = 'afterOrderSend';
    const EVENT_AFTER_ORDER_ACTIVE = 'afterOrderActive';
    const EVENT_AFTER_ORDER_PAY = 'afterOrderPay';

    public function getOrderStatus(){
        return $this->order_status;
    }

    /**
     * Aggiornamento degli order status
     */
    public function setOrderStatus($newStatus){
        $this->owner->order_status = $newStatus;
        $this->owner->save();
    }

    public function sendOrder(){
        $this->owner->setOrderStatus("sent"); 
        CustomerUser::getCurrentUser()->unsetActiveOrder();
        $this->owner->trigger(self::EVENT_AFTER_ORDER_SEND);
    }

    public function deleteOrder(){
        $this->owner->setOrderStatus("deleted"); 
        CustomerUser::getCurrentUser()->unsetActiveOrder();
    }

    public function payOrder(){
        $this->owner->setOrderStatus("paid"); 
        CustomerUser::getCurrentUser()->unsetActiveOrder();
        $this->owner->trigger(self::EVENT_AFTER_ORDER_PAY); 
    }

    public function activateOrder(){
        CustomerUser::getCurrentUser()->setActiveOrder($this->owner);  
        CustomerUser::deleteAllFilter();
        $this->owner->trigger(self::EVENT_AFTER_ORDER_ACTIVE); 
    }

    /**
     * Set order status filter
     */
    public function orderStatusFilter($orderStatus){
        $this->selectedOrderStatus = $orderStatus;
    }

    /**
     * return ['=', 'order_status', $params['order_status']]
     * ['>', 'status', 20]
     */
    public function orderStatusWhere(){
        return $filter = [
            'order_status' => $this->selectedOrderStatus,
        ];        
    }

    public function orderStatusQueryFilter($query){
        switch($this->selectedOrderStatus){
            case self::ORDER_STATUS_SENT:
                $query->andFilterWhere(
                    ['=', 'order_status', 'sent']
                );
                $query->andFilterWhere(
                    ['>', 'status', 20]
                );        
                break;
            case self::ORDER_STATUS_DELETED:
                $query->andFilterWhere(
                    ['=', 'status', 0]
                );        
                break;
            case self::ORDER_STATUS_GMV_RECEIVED:
                $query->andWhere(
                    ['!=', 'GMV_PK', 'NULL']
                );
                $query->andFilterWhere(
                    ['>', 'status', 20]
                );
                break;
            case self::ORDER_STATUS_PROCESSING:
                $query->andFilterWhere(
                    ['=', 'order_status', 'processing']
                );
                $query->andFilterWhere(
                    ['>', 'status', 20]
                );        
                break;
            case self::ORDER_STATUS_PAID:
                $query->andFilterWhere(
                    ['=', 'order_status', 'paid']
                );
                $query->andFilterWhere(
                    ['>', 'status', 20]
                );        
                break;
            case self::ORDER_STATUS_GMV_CONFIRMED:
                $query->andFilterWhere(
                    ['=', 'order_status', 'CONFERMATO']
                );      
                break;
            case self::ORDER_STATUS_GMV_SENT:
                $query->andFilterWhere(
                    ['=', 'order_status', 'SPEDITO']
                );      
                break;
            case self::ORDER_STATUS_GMV_PROCESSING:
                $query->andFilterWhere(
                    ['=', 'order_status', 'IN PREPARAZIONE']
                );      
                break;
            case self::ORDER_STATUS_GMV_READY:
                $query->andFilterWhere(
                    ['=', 'order_status', 'PRONTO PER IL RITIRO']
                );      
                break;
            case self::ORDER_STATUS_GMV_CANCEL:
                $query->andFilterWhere(
                    ['=', 'order_status', 'CANCELLATO']
                );      
                break;
            default:
                break;
        }
       
        return $query;
    }


    public static function getModelType($selectedOrderStatus){
        switch($selectedOrderStatus){
            case self::ORDER_STATUS_PROCESSING: 
            case self::ORDER_STATUS_DELETED:   
            case self::ORDER_STATUS_SENT:
            case self::ORDER_STATUS_PAID:
            case self::ORDER_STATUS_GMV_RECEIVED:
                return new OrderCustomerSearch();
                break;
            case self::ORDER_STATUS_GMV_CONFIRMED:
            case self::ORDER_STATUS_GMV_PROCESSING:
            case self::ORDER_STATUS_GMV_READY:
            case self::ORDER_STATUS_GMV_SENT:
            case self::ORDER_STATUS_GMV_CANCEL:
                return new GMVOrderSearch();
                break;
            case self::ORDER_STATUS_GMV_OTHER:
                return new GMVOrderOtherSearch();
                break;
            default:
                break;
        }
        return;
    }

    public static function getIndex($selectedOrderStatus){
        switch($selectedOrderStatus){
            case self::ORDER_STATUS_PROCESSING: 
            case self::ORDER_STATUS_DELETED:   
            case self::ORDER_STATUS_SENT:
            case self::ORDER_STATUS_PAID:
            case self::ORDER_STATUS_GMV_RECEIVED:
                return '/order-customer/index';
                break;
            case self::ORDER_STATUS_GMV_CONFIRMED:
            case self::ORDER_STATUS_GMV_PROCESSING:
            case self::ORDER_STATUS_GMV_READY:
            case self::ORDER_STATUS_GMV_SENT:
            case self::ORDER_STATUS_GMV_CANCEL:
                return '/gmv-order/index';
                break;
            case self::ORDER_STATUS_GMV_OTHER:
                return '/gmv-order-other/index';
                break;
            default:
                break;
        }
        return;
    }

    public static function getDatatableView($selectedOrderStatus){
        switch($selectedOrderStatus){
            case self::ORDER_STATUS_PROCESSING: 
            case self::ORDER_STATUS_DELETED:   
            case self::ORDER_STATUS_SENT:
            case self::ORDER_STATUS_PAID:
                return "_datatable";
                break;           
            case self::ORDER_STATUS_GMV_RECEIVED:
                return "_datatable_confirmed";
                break;           
            case self::ORDER_STATUS_GMV_CONFIRMED:
            case self::ORDER_STATUS_GMV_PROCESSING:
            case self::ORDER_STATUS_GMV_READY:
            case self::ORDER_STATUS_GMV_CANCEL:
            case self::ORDER_STATUS_GMV_SENT:
                return "_datatable_gmv";
                break;
            default:
                break;
        }
       
        return;
    }

    public static function getAllStatus(){
        return [
            self::ORDER_STATUS_PROCESSING => self::ORDER_STATUS_PROCESSING, 
            self::ORDER_STATUS_DELETED => self::ORDER_STATUS_DELETED,
            self::ORDER_STATUS_SENT => self::ORDER_STATUS_SENT,
            self::ORDER_STATUS_PAID => self::ORDER_STATUS_PAID,            
            self::ORDER_STATUS_GMV_SENT => self::ORDER_STATUS_GMV_SENT,
            self::ORDER_STATUS_GMV_CONFIRMED => self::ORDER_STATUS_GMV_CONFIRMED,
            self::ORDER_STATUS_GMV_PROCESSING => self::ORDER_STATUS_GMV_PROCESSING,
            self::ORDER_STATUS_GMV_READY => self::ORDER_STATUS_GMV_READY, 
            self::ORDER_STATUS_GMV_CANCEL => self::ORDER_STATUS_GMV_CANCEL,
            self::ORDER_STATUS_GMV_OTHER => self::ORDER_STATUS_GMV_OTHER,
        ];
    }

    public function showActionOrderItem(){
        switch($this->owner->order_status){
            case self::ORDER_STATUS_PROCESSING: 
                return true;
                break;
            case self::ORDER_STATUS_DELETED:   
            case self::ORDER_STATUS_SENT:
            case self::ORDER_STATUS_PAID:
            case self::ORDER_STATUS_GMV_RECEIVED:
            case self::ORDER_STATUS_GMV_CONFIRMED:
            case self::ORDER_STATUS_GMV_PROCESSING:
            case self::ORDER_STATUS_GMV_READY:
            case self::ORDER_STATUS_GMV_CANCEL:
            case self::ORDER_STATUS_GMV_SENT:
            case self::ORDER_STATUS_GMV_OTHER:
                return false;
                break;
            default:
                break;
        }
    }

    public function checkProcessing(){
        if($this->owner->order_status ==  self::ORDER_STATUS_PROCESSING){
            return true;
        }
        return false;
    }

    public static function getOrderAdminStatus(){
        return [
            self::ORDER_STATUS_PROCESSING => self::ORDER_STATUS_PROCESSING, 
            self::ORDER_STATUS_DELETED => self::ORDER_STATUS_DELETED,
            self::ORDER_STATUS_SENT => self::ORDER_STATUS_SENT,
            self::ORDER_STATUS_PAID => self::ORDER_STATUS_PAID
        ];
    }

}